<?php

// Enable full error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Load environment variables from the UVDesk .env file
 */
function loadUvdeskEnv($filePath)
{
    if (!file_exists($filePath)) {
        die(".env file not found at: $filePath");
    }

    $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $key = trim($key);
            $value = trim(str_replace('"', '', $value)); // Remove quotes if present
            putenv("$key=$value");
            $_ENV[$key] = $value;
            $_SERVER[$key] = $value;
        }
    }
}

// Path to UVDesk .env file (adjust if needed)
$uvdeskEnvPath = getcwd() . '/.env';

// Load UVDesk environment variables
loadUvdeskEnv($uvdeskEnvPath);

// Parse DB credentials from .env
$databaseUrl = getenv('DATABASE_URL') ?: '';
if (!$databaseUrl) {
    die("DATABASE_URL not found in UVDesk .env file!");
}

$parsedUrl = parse_url($databaseUrl);
$host = $parsedUrl['host'] ?? 'localhost';
$dbname = isset($parsedUrl['path']) ? ltrim($parsedUrl['path'], '/') : die("Database name not found!");
$user = $parsedUrl['user'] ?? 'root';
$pass = $parsedUrl['pass'] ?? '';
$port = $parsedUrl['port'] ?? '3306';

// Establish DB connection
try {
    $pdo = new PDO("mysql:host=$host;port=$port;dbname=$dbname", $user, $pass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo "Connected successfully to UVDesk database.\n";
} catch (PDOException $e) {
    die("Could not connect to the database: " . $e->getMessage());
}

/**
 * Get latest valid API token
 */
function getLatestApiToken($pdo)
{
    $query = "SELECT token FROM uv_api_access_credentials 
              WHERE is_enabled = 1 AND is_expired = 0 
              ORDER BY created_on DESC LIMIT 1";
    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result ? $result['token'] : null;
}

// Get base URL & token
$site_url = getenv('SITE_URL') ?: die("SITE_URL not found in .env file!");
$api_token = getLatestApiToken($pdo);
if (!$api_token) {
    die("No valid API token found in `uv_api_access_credentials` table!");
}

echo "Using API Token: $api_token\n";

// Fetch customers with specific support role
$query = "SELECT * FROM uv_user WHERE id IN (SELECT user_id FROM uv_user_instance WHERE supportRole_id = 4)";
$stmt = $pdo->query($query);
$customers = $stmt->fetchAll(PDO::FETCH_ASSOC);
if (!$customers) {
    die("No customers found.");
}

// Get the previous month in format: April-2025
$previousMonthYear = date('F-Y', strtotime('first day of last month'));

// Loop through each customer
foreach ($customers as $customer) {
    $ticketData = [
        'actAsType' => 'customer',
        'name' => $customer['first_name'] . ' ' . $customer['last_name'],
        'from' => $customer['email'],
        'subject' => "Monthly GST-R3 Ticket for $previousMonthYear",
        'message' => "This is your automatically generated Monthly GST-Registration ticket for $previousMonthYear.",
        'type' => 'GST-R3-001 Ensure GSTR1 is filed',
        'status' => 1, // Open
    ];

    // Create the ticket
    $ch = curl_init($site_url . '/api/v1/ticket');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer $api_token",
        "Content-Type: application/json",
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($ticketData));
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    echo "API Response: " . ($response ? $response : "No Response") . "\n";

    if ($httpCode == 200 && $response) {
        $ticket = json_decode($response, true);
        $ticketId = $ticket['ticketId'] ?? null;

        if ($ticketId) {
            $tagData = ['name' => 'GST-R3'];
            $url = "$site_url/en/api/ticket/$ticketId/tags.json";
        
            echo "Tagging URL: $url\n";
            echo "Tag Payload: " . json_encode($tagData) . "\n";
        
            $tagCh = curl_init($url);
            curl_setopt($tagCh, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($tagCh, CURLOPT_HTTPHEADER, [
                "Authorization: Bearer $api_token",
                "Content-Type: application/json",
            ]);
            curl_setopt($tagCh, CURLOPT_POSTFIELDS, json_encode($tagData));
            curl_setopt($tagCh, CURLOPT_CUSTOMREQUEST, 'POST');
            curl_setopt($tagCh, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($tagCh, CURLOPT_SSL_VERIFYHOST, false);
        
            $tagResponse = curl_exec($tagCh);
            $tagCurlError = curl_error($tagCh);
            $tagHttpCode = curl_getinfo($tagCh, CURLINFO_HTTP_CODE);
            curl_close($tagCh);
        
            if ($tagResponse === false) {
                echo "❌ cURL Error while tagging: $tagCurlError\n";
            }
        
            if ($tagHttpCode == 200) {
                echo "✅ Tag 'GST-R3' added to ticket ID $ticketId successfully.\n";
            } else {
                echo "❌ Failed to tag ticket ID $ticketId. HTTP: $tagHttpCode\n";
                echo "Response: $tagResponse\n";
            }
        }

        echo "Ticket created successfully for " . $customer['email'] . "\n";
    } else {
        echo "Failed to create ticket for " . $customer['email'] . "\n";
        echo "HTTP Code: $httpCode\n";
        echo "Response: " . ($response ? $response : "No Response") . "\n";
    }
}
