<?php

// Enable full error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Load environment variables from the UVDesk .env file
 */
function loadUvdeskEnv($filePath)
{
    if (!file_exists($filePath)) {
        die(".env file not found at: $filePath");
    }

    $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $key = trim($key);
            $value = trim(str_replace('"', '', $value)); // Remove quotes if present
            putenv("$key=$value");
            $_ENV[$key] = $value;
            $_SERVER[$key] = $value;
        }
    }
}

// // Path to UVDesk .env file (Update this path as per your UVDesk installation)
// $uvdeskEnvPath = '/var/www/html/uvdesk/.env'; // For Linux servers
 $uvdeskEnvPath = getcwd() . '/.env'; // For Windows XAMPP

// // Load UVDesk environment variables
loadUvdeskEnv($uvdeskEnvPath);

// Get DB credentials from UVDesk .env file
$databaseUrl = getenv('DATABASE_URL') ?: '';

if (!$databaseUrl) {
    die("DATABASE_URL not found in UVDesk .env file!");
}

// Parse DATABASE_URL (mysql://user:password@host:port/dbname)
$parsedUrl = parse_url($databaseUrl);

$host = $parsedUrl['host'] ?? 'localhost';
$dbname = isset($parsedUrl['path']) ? ltrim($parsedUrl['path'], '/') : die("Database name not found!");
$user = $parsedUrl['user'] ?? 'root';
$pass = $parsedUrl['pass'] ?? '';
$port = $parsedUrl['port'] ?? '3306';


// Helpdesk API credentials
// $helpdesk_url = 'http://localhost/public';
// $api_token = 'Z0JR6UUV7ERB1XRECVDUJWCWYUB0RNTZN067CRECH0XUD5FCMHMT7COKI8VFEEVQ';

// Establish database connection
try {
    $dsn = "mysql:host=$host;port=$port;dbname=$dbname";
    $pdo = new PDO($dsn, $user, $pass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo "Connected successfully to UVDesk database.\n";
} catch (PDOException $e) {
    die("Could not connect to the database: " . $e->getMessage());
}

/**
 * Fetch API Token from `uv_api_access_credentials`
 */
function getLatestApiToken($pdo)
{
    $query = "SELECT token FROM uv_api_access_credentials 
              WHERE is_enabled = 1 AND is_expired = 0 
              ORDER BY created_on DESC LIMIT 1";

    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    return $result ? $result['token'] : null;
}

// Get Site API URL from .env
$site_url = getenv('SITE_URL') ?: die("SITE_URL not found in .env file!");

// Get API Token from Database
$api_token = getLatestApiToken($pdo);

if (!$api_token) {
    die("No valid API token found in `uv_api_access_credentials` table!");
}

echo "Using API Token: $api_token\n";

// Get today's date
$today = (new DateTime())->format('Y-m-d');

// Get the custom field IDs
$customFieldNames = ['Reminder Date', 'Prior Day', 'Message'];
$customFieldIds = [];

try {
    foreach ($customFieldNames as $fieldName) {
        $query = "SELECT id FROM uv_pkg_uvdesk_form_component_custom_fields WHERE name = :name";
        $stmt = $pdo->prepare($query);
        $stmt->execute(['name' => $fieldName]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($result) {
            $customFieldIds[$fieldName] = $result['id'];
        }
    }

    if (count($customFieldIds) == 3) {
        // Reminder Notification Logic
        $reminderDateId = $customFieldIds['Reminder Date'];
        $priorDayId = $customFieldIds['Prior Day'];

        $query = "SELECT ticket_id, 
                 (SELECT value FROM uv_pkg_uvdesk_form_component_ticket_custom_fields_values 
                  WHERE custom_field_id = :reminder_date_id AND ticket_id = tcf.ticket_id 
                  AND TRIM(value) <> '' LIMIT 1) as reminder_date,
                 (SELECT value FROM uv_pkg_uvdesk_form_component_ticket_custom_fields_values 
                  WHERE custom_field_id = :prior_day_id AND ticket_id = tcf.ticket_id 
                  AND TRIM(value) <> '' LIMIT 1) as prior_day
          FROM uv_pkg_uvdesk_form_component_ticket_custom_fields_values tcf
          WHERE custom_field_id = :reminder_date_id 
          AND TRIM(value) <> ''";

        $stmt = $pdo->prepare($query);
        $stmt->bindParam(':reminder_date_id', $reminderDateId, PDO::PARAM_INT);
        $stmt->bindParam(':prior_day_id', $priorDayId, PDO::PARAM_INT);
        $stmt->execute([
            'reminder_date_id' => $reminderDateId,
            'prior_day_id' => $priorDayId
        ]);

        $tickets = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($tickets as $ticket) {
            $ticketId = $ticket['ticket_id'];
            $reminderDateString = trim($ticket['reminder_date'], '"');
            if ($reminderDateString==""){
                continue;
            }
            $reminderDate = new DateTime($reminderDateString);
            $priorDay = (int)trim($ticket['prior_day'] ?? '', '"');
            $todayDate = (new DateTime())->setTime(0, 0);
            $reminderDate = $reminderDate->setTime(0, 0);
            $interval = $todayDate->diff($reminderDate)->days;

            if ($interval == $priorDay) {
                // Fetch customer details
                $query = "SELECT u.email, u.first_name, u.last_name 
                          FROM uv_user u 
                          JOIN uv_ticket t ON u.id = t.customer_id 
                          WHERE t.id = :ticket_id";
                $stmt = $pdo->prepare($query);
                $stmt->execute(['ticket_id' => $ticketId]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);

                // Fetch message content from 'Message' custom field
                $query = "SELECT value as message FROM uv_pkg_uvdesk_form_component_ticket_custom_fields_values 
                          WHERE ticket_id = :ticket_id AND custom_field_id = :message_field_id";
                $stmt = $pdo->prepare($query);
                $stmt->execute([
                    'ticket_id' => $ticketId,
                    'message_field_id' => $customFieldIds['Message']
                ]);
                $messageFieldValue = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($user && $messageFieldValue) {
                    // Prepare API data
                    $message = str_replace(
                        ['{targetDate}', '{firstName}', '{lastName}', '{email}', '{ticketLink}'], 
                        [$reminderDate->format('Y-m-d'), $user['first_name'], $user['last_name'], $user['email'], "<a href=\"{$site_url}/member/ticket/view/{$ticketId}\">ticket</a>"], 
                        $messageFieldValue['message']
                    );

                    $data = [
                        'message' => $message,
                        'actAsType' => 'agent',
                        'actAsEmail' => 'mohamed.shimil@dranzo.com',
                        'name' => "{$user['first_name']} {$user['last_name']}",
                        'subject' => 'Reminder',
                        'from' => $user['email']
                    ];

                    // Create new ticket via API using cURL
                    $url = $site_url . '/api/v1/ticket';
                    $ch = curl_init($url);

                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_HTTPHEADER, [
                        "Content-Type: application/json",
                        "Authorization: Basic " . base64_encode($api_token)
                    ]);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));

                    // Execute the request and capture the response
                    $result = curl_exec($ch);

                    // Check for errors
                    if (!curl_errno($ch)) {
                        $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                        if ($http_status == 200) {
                            // Optionally log the successful result
                        }
                    }

                    // Close the cURL session
                    curl_close($ch);
                }
            }
        }
    }

    // Due Date Notification Logic
    // Fetch tickets where the due date is today
    $query = "SELECT 
            t.id as id,
                tt.code AS ticket_type_name
            FROM 
                uv_ticket t
            INNER JOIN 
                uv_ticket_type tt
            ON 
                t.type_id = tt.id WHERE CAST(due_date AS DATE) = :today";
    $stmt = $pdo->prepare($query);
    $stmt->execute(['today' => $today]);
    $tickets = $stmt->fetchAll(PDO::FETCH_ASSOC);

    foreach ($tickets as $ticket) {
        $ticketId = $ticket['id'];
        $Ntype = $ticket['ticket_type_name'];

        // Fetch the user with id=1
        $query = "SELECT email, first_name, last_name FROM uv_user WHERE id = 1";
        $stmt = $pdo->prepare($query);
        $stmt->execute();
        $adminUser = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($adminUser) {
            $email = $adminUser['email'];
            $firstName = $adminUser['first_name'];
            $lastName = $adminUser['last_name'];

            // Prepare data for creating a notification ticket
            $data = [
                'actAsType' => 'agent',
                'actAsEmail' => $email,
                'name' => "$firstName $lastName",
                'subject' => 'Due Date Reached Notification',
                'type' => 'N - '.$Ntype,
                'from' => $email,
                'message' => "The due date for ticket ID {$ticketId} has been reached. Please take the necessary actions."
            ];

            // Create new notification ticket via API using cURL
            $url = $site_url . '/api/v1/ticket';
            $ch = curl_init($url);

            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Content-Type: application/json",
                "Authorization: Basic " . base64_encode($api_token)
            ]);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));

            // Execute the request and capture the response
            $result = curl_exec($ch);

            // Check for errors
            if (!curl_errno($ch)) {
                $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                if ($http_status == 200) {
                    // Optionally log the successful result
                }
            }

            // Close the cURL session
            curl_close($ch);
        }
    }
    }
 catch (Exception $e) {
    // Handle exceptions
}

?>
